;-----------------------------------------------------------------------------------------------------------
; Proyecto: Pesebre - Tarjeta de control Casas
; Autor: Gabriel Rapetti (Inventable)
; Versin 1
; Fecha: 30/10/2014
; Micro: PIC12F629
; Clock: oscilador interno 4MHz
; ----------------------------------------------------------------------------------------------------------

; -----------------------------------------------------------------------------------------------------------
; Licencia Creative Commons BY-NC-SA (attribution - Non Commercial - Share Alike)
; El proyecto puede ser hecho y modificado por los lectores para uso privado y sin fines comerciales
; Es necesario indicar en eventuales publicaciones el autor (Gabriel Rapetti o Inventable.eu)
; y se pueden hacer obras derivadas. 
;-----------------------------------------------------------------------------------------------------------
	ERRORLEVEL -302	   	
	LIST 	P = 12F629, r=DEC, n=66
    INCLUDE         "P12F629.INC"

    __CONFIG _CP_ON & _CPD_OFF & _BODEN_ON & _MCLRE_OFF & _WDT_ON & _PWRTE_ON & _INTRC_OSC_NOCLKOUT

;-----------------------------------------------------------------------------------------------------------
; Pines del micro-controlador
; ===========================
;-----------------------------------------------------------------------------------------------------------

#define		p_casa1	GPIO,0				; pin salida casa 1 
#define		p_casa2	GPIO,1				; pin salida casa 2
#define		p_casa3	GPIO,2				; pin salida casa 3
#define 	p_casa4	GPIO,5				; pin salida casa 4
#define 	p_adc	GPIO,4				; pin potencimetro para el retardo del encendido 
#define 	p_sinc	GPIO,3				; pin de entrada sincro para activar las luces de noche

#define 	IOMap_normal	B'001000'	; mapa entradas/salidas del micro trabajando normalmente
#define 	IOMap_adc		B'011000'	; mapa entradas/salidas del micro cuando se lee el potencimetro

;-----------------------------------------------------------------------------------------------------------
; Variables 
; ================
; de un byte (valor entre 0 y 255) 
;-----------------------------------------------------------------------------------------------------------

	cblock     20H     ; Dummy Origin

		casa1							; variable intensidad canal casa 1
		casa2							; variable intensidad canal casa 2
		casa3							; variable intensidad canal casa 3
		casa4							; variable intensidad canal casa 4

		timerReg						; variable usada para la temporizacin de los destellos  

		CountDelay1						; variable usada como contador para funciones de temporizacin  
		CountDelay2						; variable usada como contador para funciones de temporizacin
		CountDelay3						; variable usada como contador para funciones de temporizacin

		CountAdc						; contiene la lectura del potencimetro  
		RegPwm							; registro que permite la regulacin de la intensidad de los canales PWM
		atenuacion
		RandomReg						; variable donde se memoriza el valor casual (random) generado

		AntiRebote
		 
	endc


;-----------------------------------------------------------------------------------------------------------
; MACRO cambio pagina de la RAM del micro
; =======================================
;-----------------------------------------------------------------------------------------------------------

PAG_RAM0	MACRO
			bcf	STATUS,RP0
			ENDM

PAG_RAM1	MACRO
			bsf	STATUS,RP0
			ENDM

;-----------------------------------------------------------------------------------------------------------
; MACRO Channel PWM Refresh 
; ==========================
; Sirve para crear la modulacin PWM que controla la intensidad luminosa.
; Los parmetros de entrada de la macro son:
;    CHANREG = registro del canal luminoso (luz_dia, luz_noche o luz_trans)
;    PWMCOUNT: contador que explora el duty cycle (es necesario incrementarlo continuamente entre 0 y 255)
;    PORT,PIN: el puerto y el pin de salida del canal (por ejemplo GPIO,0)         
;-----------------------------------------------------------------------------------------------------------
PwmChRefr	MACRO	CHANREG,PWMCOUNT,PORT,PIN
			movf	CHANREG,W				; del canal verde
			subwf	PWMCOUNT,W
			btfss	STATUS,C
			bcf		PORT,PIN
			btfsc	STATUS,C
			bsf		PORT,PIN
			ENDM

;-----------------------------------------------------------------------------------------------------------
; Vector de arranque del micro
; ============================
;-----------------------------------------------------------------------------------------------------------
    		ORG     0
			PAG_RAM1
			call 	3FFh 			; Ajusta el oscilador a 4MHz
			movwf 	OSCCAL 			; usando el valor que se encuentra en 3FFh
			PAG_RAM0
			goto	main

;-----------------------------------------------------------------------------------------------------------
;					SUBRUTINAS
;-----------------------------------------------------------------------------------------------------------

			ORG	10h					; beginning of program Eprom

;----------------------------------------------------------------------------------------------------------- 
; lectura del potencimetro (ADC software)
; ========================================
; Permite la lectura de un potencimetro con un micro desprovisto de ADC (convertidor analgico / digital) 
; Para ello se mide el tiempo de descarga de un capacitor de 10nF a travs del potencimetro. 
; El pin conectado al potencimetro/capacitor se usa como salida (para descargar el capacitor) y como entrada
; (para saber cuando el capacitor se carga). La lectura del potencimetro se memoriza en la variable "CountAdc"    
;----------------------------------------------------------------------------------------------------------- 		
ADC		
		movlw	1
		movwf	CountAdc			; Reset del contador
		PAG_RAM1
		movlw   IOMap_adc    		; Pone el pin ADC como entrada para leer el potencimetro 
 		movwf	TRISIO 
		PAG_RAM0  
 	
cicloADC
		clrwdt
		btfss	p_adc				; Controla si el capacitor se carg 
		goto	finADC				; si, entonces termina    
		incfsz 	CountAdc,f			; no, entonces incrementa el contador y salta al ciclo
		goto	cicloADC
		decf	CountAdc,f			; si el contador llega al mximo, termina 
finADC
		PAG_RAM1
		movlw   IOMap_normal		; Pone el pin ADC como salida para permitir
 		movwf	TRISIO 				; que el capacitor se descargue
		PAG_RAM0
		bsf		p_adc				; descarga el capacitor y termina
		retlw	0

;-----------------------------------------------------------------------------------------------------------
; Funcin para la temporizacin antes que se activen las casas  
; =================================================================
; La duracin se puede regular pasando un valor entre 1 y 255 en W
;-----------------------------------------------------------------------------------------------------------
DlyTimer	
		movwf	CountDelay3			; X3 multiplicador
DlyJp0
		movlw	255					; X2 multiplicador
		movwf	CountDelay2

DlyJp1	clrwdt
		movlw	255					; X1 multiplicador
		movwf	CountDelay1

DlyJp2	nop							; Ciclo principal
		nop
		decfsz	CountDelay1,F
		goto	DlyJp2

		decfsz	CountDelay2,F
		goto 	DlyJp1

		decfsz	CountDelay3,F
		goto 	DlyJp0

		retlw	0

;-----------------------------------------------------------------------------------------------------------
; Actualiza los 4 canales luminosos con PWM 
; ============================================
; usa la macro PwmChRefr
;-----------------------------------------------------------------------------------------------------------
PwmRefresh
		PwmChRefr casa1,RegPwm,p_casa1
		PwmChRefr casa2,RegPwm,p_casa2
		PwmChRefr casa3,RegPwm,p_casa3
		PwmChRefr casa4,RegPwm,p_casa4
		retlw 0

;-----------------------------------------------------------------------------------------------------------
; Generador de nmeros pseudo casuales 
; =====================================
;-----------------------------------------------------------------------------------------------------------
rand
		rlf		RandomReg,W
		rlf		RandomReg,W
		btfsc	RandomReg,4
		xorlw	1
		btfsc	RandomReg,5
		xorlw	1
		btfsc	RandomReg,3
		xorlw	1
		movwf	RandomReg

		movf	atenuacion,W
		subwf	RandomReg,W
		btfss	STATUS,C
		goto	randOffSet	
		return

randOffSet
		movlw	100
		subwf	atenuacion,w
		btfss	STATUS,C
		retlw 	50
		retlw	0

;-----------------------------------------------------------------------------------------------------------
;					MAIN
;-----------------------------------------------------------------------------------------------------------
main    	
			PAG_RAM1
			movlw   IOMap_normal    		
			movwf	TRISIO				; Set de los puertos como entradas o salidas 

			movlw	B'11100011'			; RB pull-up on. Prscaler %128 in Watch-Dog (max.time)
			movwf	OPTION_REG			; 
		
			PAG_RAM0

			movlw	B'00000111'			; Set de los puertos como entradas/salidas solo digitales
			movwf	CMCON

			bcf	INTCON,GIE				; Ningn interrupt global activado
			bcf	INTCON,PEIE				; Ningn interrupt perifrico activado

			clrf	casa1				; luz casa 1 apagada
			clrf	casa2				; luz casa 2 apagada
			clrf	casa3				; luz casa 3 apagada
			clrf	casa4				; luz casa 4 apagada
		 
			bsf		p_adc				; Descarga el capacitor para la lectura del potencimetro 
			
;-----------------------------------------------------------------------------------------------------------
;  Da
; =====
;-----------------------------------------------------------------------------------------------------------
Dia			
			bsf		p_casa1				; pin de salida del canal casa 1 apagada  
			bsf		p_casa2				; pin de salida del canal casa 2 apagada
			bsf		p_casa3				; pin de salida del canal casa 3 apagada
			bsf		p_casa4				; pin de salida del canal casa 4 apagada
			clrf	AntiRebote
DiaJp1	
			clrwdt
			btfss	p_sinc				; Controla el estado del circuito Da / noche 
			goto	Dia
			incfsz	AntiRebote,f		; Repite el control muchas veces (para eliminar posibles rebotes) 
			goto	DiaJp1				; Si circuito Da/noche se encuentra en el da salta a DiaJp1  

;-----------------------------------------------------------------------------------------------------------
; Atardecer
; =========
;-----------------------------------------------------------------------------------------------------------
Atardecer
			call	ADC					; Lee el potencimetro
			movf	CountAdc,w			; Pasa el valor del potencimetro a la sub de temporizacin 
			call	DlyTimer			; Llamada a la sub de retardo encendido casas 
			movlw	255					; La variable "atenuacion" sirve per encender ls luces
			movwf	atenuacion			; gradualmente cuando llega la noche
;-----------------------------------------------------------------------------------------------------------
; Noche
; =====
;-----------------------------------------------------------------------------------------------------------
Noche
			clrwdt
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f			; 255 veces (genera la onda pwm)
			goto	Noche
			decfsz	timerReg,f			; Segundo loop para disminuir la velocidad 
			goto	Noche				; de parpadeo de las luces

			call	rand				; Genera un valor casual de luz para la casa 1
			movwf	casa1
 			call	rand				; Genera un valor casual de luz para la casa 2
			movwf	casa2
			call	rand				; Genera un valor casual de luz para la casa 3
			movwf	casa3
			call	rand				; Genera un valor casual de luz para la casa 4
			movwf	casa4

			movlw	10					; Repone el valor de la variable que regula la velocidad
			movwf	timerReg			; de parpadeo de las luces

			movlw	100					; Controla si atenuacin lleg al valor de 100
			subwf	atenuacion,w		; Cuando atenuacin llega a 100 la intensidad de
			btfsc	STATUS,C			; de las luces ser mxima
			decf	atenuacion,f		; Si no es as, disminuye "atenuacion" de 1	

			clrf	AntiRebote

ControlFineNoche
			clrwdt	
			btfsc	p_sinc				; Controla el estado del circuito "Da / noche" 
			goto	Noche				; Si est todava en la noche salta a "Noche"
			incfsz	AntiRebote,f
			goto	ControlFineNoche

;-----------------------------------------------------------------------------------------------------------
;  Alba
; ======
; (similar a noche pero con las luces que disminuyen progresivamente)  
;-----------------------------------------------------------------------------------------------------------
Alba
			clrwdt
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f			; 255 veces (genera la onda pwm)
			goto	Alba
			decfsz	timerReg,f			; Segundo loop para disminuir la velocidad
			goto	Alba				; de parpadeo de las luces

			call	rand				; Genera un valor casual de luz para la casa 1
			movwf	casa1
 			call	rand				; Genera un valor casual de luz para la casa 2
			movwf	casa2
			call	rand				; Genera un valor casual de luz para la casa 3
			movwf	casa3
			call	rand				; Genera un valor casual de luz para la casa 4
			movwf	casa4

			movlw	10					; Repone el valor de la variable que regula la velocidad
			movwf	timerReg			; de parpadeo de las luces

			incfsz	atenuacion,f		; disminuye progresivamente la intensidad de la luces 
			goto	Alba				; y vuelve al ciclo 
			goto	Dia					; Cuando la intensidad es muy baja termina el alba 
										; saltando a la seccin "Da"

;***********************************************************************************************
; Service Interrupt: no se usa
;***********************************************************************************************
service_int
		retfie
;***********************************************************************************************
		END
