;-----------------------------------------------------------------------------------------------------------
; Proyecto: Pesebre - Tarjeta de control Da y noche
; Autor: Gabriel Rapetti (Inventable.eu)
; Fecha: 30/10/2014
; Micro: PIC12F629
; Clock: oscilador interno 4MHz
;-----------------------------------------------------------------------------------------------------------

; -----------------------------------------------------------------------------------------------------------
; Licencia Creative Commons BY-NC-SA (attribution - Non Commercial - Share Alike)
; El proyecto puede ser hecho y modificado por los lectores para uso privado y sin fines comerciales
; Es necesario indicar en eventuales publicaciones el autor (Gabriel Rapetti o Inventable.eu)
; y se pueden hacer obras derivadas. 
;-----------------------------------------------------------------------------------------------------------
	ERRORLEVEL -302	   	
	LIST 	P = 12F629, r=DEC, n=66
    INCLUDE         "P12F629.INC"

    __CONFIG _CP_ON & _CPD_OFF & _BODEN_ON & _MCLRE_OFF & _WDT_ON & _PWRTE_ON & _INTRC_OSC_NOCLKOUT

;-----------------------------------------------------------------------------------------------------------
; Pines del micro-controlador
; ===========================
;-----------------------------------------------------------------------------------------------------------

#define		p_dia	GPIO,0				; pin luz da 
#define		p_noche	GPIO,1				; pin luz noche 
#define		p_trans	GPIO,2				; pin luz transicin (atardecer y alba)
#define 	p_adc	GPIO,4				; pin del potencimetro (tiempo ciclo)
#define 	p_sinc	GPIO,5				; pin salida sincro para activar las luces de las casas 

#define 	IOMap_normal	B'001000'	; mapa entradas/salidas del micro trabajando normalmente
#define 	IOMap_adc		B'011000'	; mapa entradas/salidas del micro cuando se lee el potencimetro


;-----------------------------------------------------------------------------------------------------------
; Variables 
; ================
; de un byte (valor entre 0 y 255) 
;-----------------------------------------------------------------------------------------------------------

	cblock     20H     ; Dummy Origin

		luz_dia							; variable intensidad canal luz de da (blanco / amarillo)		 
		luz_noche				 		; variable intensidad canal luz de noche (azul)		 
		luz_trans						; variable intensidad canal luz transicin (rojo / naranja)

		timerReg						; variable usada para la temporizacin del da y de la noche  

		CountDelay1						; variable usada como contador para obtener los tiempos da / noche  
		CountDelay2						; variable usada como contador para obtener los tiempos da / noche
		CountDelay3						; variable usada como contador para obtener los tiempos da / noche

		CountAdc						; contiene la lectura del potencimetro  
		RegPwm							; registro que permite la regulacin de la intensidad de los canales PWM
		DlyTrans						; variable usada para la temporizacin de la transicin (atardecer y alba)
			 
	endc


;-----------------------------------------------------------------------------------------------------------
; MACRO cambio pagina de la RAM del micro
; =======================================
;-----------------------------------------------------------------------------------------------------------

PAG_RAM0	MACRO
			bcf	STATUS,RP0
			ENDM

PAG_RAM1	MACRO
			bsf	STATUS,RP0
			ENDM

;-----------------------------------------------------------------------------------------------------------
; MACRO Channel PWM Refresh 
; ==========================
; Sirve para crear la modulacin PWM que controla la intensidad luminosa.
; Los parmetros de entrada de la macro son:
;    CHANREG = registro del canal luminoso (luz_dia, luz_noche o luz_trans)
;    PWMCOUNT: contador que explora el duty cycle (es necesario incrementarlo continuamente entre 0 y 255)
;    PORT,PIN: el puerto y el pin de salida del canal (por ejemplo GPIO,0)         
;-----------------------------------------------------------------------------------------------------------
PwmChRefr	MACRO	CHANREG,PWMCOUNT,PORT,PIN
			movf	CHANREG,W				; del canal verde
			subwf	PWMCOUNT,W
			btfss	STATUS,C
			bcf		PORT,PIN
			btfsc	STATUS,C
			bsf		PORT,PIN
			ENDM

;-----------------------------------------------------------------------------------------------------------
; Vector de arranque del micro
; ============================
;-----------------------------------------------------------------------------------------------------------
    		ORG     0
			PAG_RAM1
			call 	3FFh 			; Ajusta el oscilador a 4MHz
			movwf 	OSCCAL 			; usando el valor que se encuentra en 3FFh
			PAG_RAM0
			goto	main

;-----------------------------------------------------------------------------------------------------------
;					SUBRUTINAS
;-----------------------------------------------------------------------------------------------------------

			ORG	10h					; beginning of program Eprom

;----------------------------------------------------------------------------------------------------------- 
; lectura del potencimetro (ADC software)
; ========================================
; Permite la lectura de un potencimetro con un micro desprovisto de ADC (convertidor analgico / digital) 
; Para ello se mide el tiempo de descarga de un capacitor de 10nF a travs del potencimetro. 
; El pin conectado al potencimetro/capacitor se usa como salida (para descargar el capacitor) y como entrada
; (para saber cuando el capacitor se carga). La lectura del potencimetro se memoriza en la variable "CountAdc"    
;----------------------------------------------------------------------------------------------------------- 		
ADC		
		movlw	1
		movwf	CountAdc			; Reset del contador
		PAG_RAM1
		movlw   IOMap_adc    		; Pone el pin ADC como entrada para leer el potencimetro 
 		movwf	TRISIO 
		PAG_RAM0  
 	
cicloADC
		clrwdt
		btfss	p_adc				; Controla si el capacitor se carg 
		goto	finADC				; si, entonces termina    
		incfsz 	CountAdc,f			; no, entonces incrementa el contador y salta al ciclo
		goto	cicloADC
		decf	CountAdc,f			; si el contador llega al mximo, termina 
finADC
		PAG_RAM1
		movlw   IOMap_normal		; Pone el pin ADC como salida para permitir
 		movwf	TRISIO 				; que el capacitor se descargue
		PAG_RAM0
		bsf		p_adc				; descarga el capcitor y termina
		retlw	0

;-----------------------------------------------------------------------------------------------------------
; Funcin para la temporizacin del da y la noche  
; ================================================
; La duracin se puede regular pasando un valor entre 1 y 255 en W
;-----------------------------------------------------------------------------------------------------------
DlyTimer	
		movwf	CountDelay3			; X3 multiplicador
DlyJp0
		movlw	50					; X2 multiplicador
		movwf	CountDelay2

DlyJp1	clrwdt
		movlw	50					; X1 multiplicador
		movwf	CountDelay1

DlyJp2	nop							; Ciclo principal
		nop
		decfsz	CountDelay1,F
		goto	DlyJp2

		decfsz	CountDelay2,F
		goto 	DlyJp1

		decfsz	CountDelay3,F
		goto 	DlyJp0

		retlw	0

;-----------------------------------------------------------------------------------------------------------
; Actualiza los 3 canales luminosos con PWM 
; ============================================
; usa la macro PwmChRefr
;-----------------------------------------------------------------------------------------------------------
PwmRefresh
		PwmChRefr luz_dia,RegPwm,p_dia
		PwmChRefr luz_noche,RegPwm,p_noche
		PwmChRefr luz_trans,RegPwm,p_trans
		retlw 0

;-----------------------------------------------------------------------------------------------------------
;					MAIN
;-----------------------------------------------------------------------------------------------------------
main    	
			PAG_RAM1
			movlw   IOMap_normal    		
			movwf	TRISIO				; Set de los puertos como entradas o salidas 

			movlw	B'11100011'			; RB pull-up on. Prscaler %128 in Watch-Dog (max.time)
			movwf	OPTION_REG			; 
		
			PAG_RAM0

			movlw	B'00000111'			; Set de los puertos como entradas/salidas solo digitales
			movwf	CMCON

			bcf	INTCON,GIE				; Ningun interrupt global activado
			bcf	INTCON,PEIE				; Ningun interrupt perifrico activado

			clrf	luz_dia				; registro intensidad luz_dia a 0 (luz apagada)		 
			clrf	luz_noche			; registro intensidad luz_noche	(luz apagada)		 	
			clrf	luz_trans			; registro intensidad luz_transicin (luz apagada)			 
			bsf		p_adc				; Descarga el capacitor para la lectura del potencimetro
			
;-----------------------------------------------------------------------------------------------------------
;				CICLO DE TRABAJO PRINCIPAL
;-----------------------------------------------------------------------------------------------------------
Ciclo
;-----------------------------------------------------------------------------------------------------------
;					DIA
;-----------------------------------------------------------------------------------------------------------
Dia
			movlw	10					; Valor temporizacin del da. Si se desea un tiempo ms 
			movwf	timerReg			; largo del que el potencimertro permite, aumentar este valor
			bcf		p_dia				; Enciende la luz del da
			bsf		p_noche				; Apaga la luz de la noche 
			bsf		p_trans				; Apaga la luz de la transicin (atardecer / alba) 
			bcf		p_sinc				; A nivel bajo la salida SINC (til para las luces de las casas) 
DiaJp1			
			clrwdt						; Actualiza el watchdog (para resetear el micro automat. si se bloca  
			call	ADC					; Lee el potencimetro para calcular la duracin del da
			movf	CountAdc,w			; Pasa el valor del potencimetro a la sub de temporizacin 
			call	DlyTimer			; 
			decfsz	timerReg,f			; Decrementa la variable de temporizacin repitiendo el bucle  
			goto	DiaJp1				; hasta que llega a 0.

;-----------------------------------------------------------------------------------------------------------
;					ATARDECER
;-----------------------------------------------------------------------------------------------------------
Atardecer
			bsf		p_sinc				; Activa la salida SINC (luces de las casas y otros efectos)   
			clrf	timerReg
			movlw	255
			movwf	luz_dia				; luz del da a mxima intensidad
			clrf	luz_noche			; luz de noche apagada		 
			clrf	luz_trans			; luz de transicin apgada
			clrf	RegPwm				; variable PWM a 0 	

; Divisin del valor del potencimetro 
; por rotacin a la derecha para
; hacer ms veloz el atardecer
; ------------------------------------- 
			bcf		STATUS,C			; reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; divisin x 2 (rotacin a la derecha)
			bcf		STATUS,C			; reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; otra divisin x 2 (rotacin a la derecha)
			bcf		STATUS,C			; reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; ltima divisin x 2 (rotacin a la derecha)
			movf	CountAdc,w			; copia el valore dividido de CountAdc en DlyTrans
			movwf	DlyTrans

; Cross-fade entre la luz del da y   
; la luz de la transicin
; -------------------------------------
AtardJp1
			clrwdt
			bsf		p_sinc
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f			; 255 veces (genera la onda pwm)
			goto	AtardJp1
			decfsz	DlyTrans,f			; Segundo loop para disminuir la velocidad
			goto	AtardJp1			; de transicin

			movf	CountAdc,w
			movwf	DlyTrans
			decf	luz_dia,f			; Disminuye luz de da
			incfsz	luz_trans,w
			incf	luz_trans,f			; Aumenta la luz del atardecer
			incfsz	timerReg,f
			goto	AtardJp1
			clrf	luz_dia				; apaga completamente luz del da
			movf	CountAdc,w
			movwf	DlyTrans		

; Cross-fade entre la luz de la 
; transicin y la luz de la noche
; -------------------------------------
AtardJp2
			clrwdt
			bsf		p_sinc				; Activa la salida SINC para encender las luces de las casas
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f
			goto	AtardJp2
			decfsz	DlyTrans,f			; Segundo loop para disminuir la velocidad
			goto	AtardJp2			; de transicin

			movf	CountAdc,w
			movwf	DlyTrans
			decf	luz_trans,f			; Disminuye la luz del atardecer
			incf	luz_noche,f			; Incrementa la luz de la noche
			incfsz	timerReg,f
			goto	AtardJp2

;-----------------------------------------------------------------------------------------------------------
;					NOCHE
;-----------------------------------------------------------------------------------------------------------
Noche
			movlw	20
			movwf	timerReg
			bsf		p_dia				; luz del da apagada
			bcf		p_noche				; luz de la noche encendida
			bsf		p_trans				; luz del atardecer / alba apagada  
			bsf		p_sinc				; salida para las luces de las casas activa
NocheJp1			
			clrwdt
			call	ADC					; Lee el potencimetro para calcular la duracin de la noche
			movf	CountAdc,w
			call	DlyTimer			; Llamado al ciclo de espera de la noche 
			decfsz	timerReg,f
			goto	NocheJp1	

;-----------------------------------------------------------------------------------------------------------
;					AMANECER
;-----------------------------------------------------------------------------------------------------------
Alba		
			bcf		p_sinc				; Desactiva la salida SINC para apagar las luces de las casas 
			clrf	timerReg
			movlw	255
			movwf	luz_noche
			clrf	luz_dia				; Registro intensidad luz_dia		 
			clrf	luz_trans			; Registro intensidad luz_transicin
			clrf	RegPwm				; Registro para generar el PWM

; Divisin del valor del potencimetro 
; por rotacin a la derecha para
; hacer ms veloz el atardecer
; ------------------------------------- 
			bcf		STATUS,C			; Reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; Divisin x 2 (rotacin a la derecha)
			bcf		STATUS,C			; Reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; Otra divisin x 2 (rotacin a la derecha)
			bcf		STATUS,C			; Reset del bit de carry antes de la rotacin
			rrf		CountAdc,f			; ltima divisin x 2 (rotacin a la derecha)
			movf	CountAdc,w			; Copia el valore dividido de CountAdc en DlyTrans
			movwf	DlyTrans

; Cross-fade entre la luz de la noche   
; y la luz de la transicin
; -------------------------------------
AlbaJp1
			clrwdt
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f			; 255 veces (genera la onda pwm)
			goto	AlbaJp1
			decfsz	DlyTrans,f			; Segundo loop para disminuir la velocidad
			goto	AlbaJp1				; de transicin

			movf	CountAdc,w
			movwf	DlyTrans
			decf	luz_noche,f			; Disminuye luz de la noche
			incfsz	luz_trans,w			; Aumenta la luz del alba
			incf	luz_trans,f	
			incfsz	timerReg,f
			goto	AlbaJp1
			clrf	luz_noche			; Luz noche apgada
			movf	CountAdc,w
			movwf	DlyTrans		

; Cross-fade entre la luz de la    
; transicin y la luz del da 
; -------------------------------------
AlbaJp2
			clrwdt
			call	PwmRefresh			; Llama la sub que actualiza la luces
			incfsz	RegPwm,f			; 255 veces (genera la onda pwm)
			goto	AlbaJp2
			decfsz	DlyTrans,f			; Segundo loop para disminuir la velocidad
			goto	AlbaJp2				; de transicin

			movf	CountAdc,w
			movwf	DlyTrans
			decf	luz_trans,f			; Disminuye luz del alba
			incf	luz_dia,f			; Aumenta la luz del da
			incfsz	timerReg,f
			goto	AlbaJp2

; Una vez que el alba termina se vuelve
; al da completando el ciclo
; -------------------------------------
			goto	Ciclo

;***********************************************************************************************
; Service Interrupt: no se usa
;***********************************************************************************************
service_int
		retfie
;***********************************************************************************************
		END
